#Include Once "words_list/words_list.bas"
#Include Once "variables.bas"

Enum errorType
  NONE = 0
  BAD_VAR_NAME = 1
  KEYWORD_VAR_NAME = 2
  DUPLICATED_DEF = 3
End Enum

Type interpreter
  Public:
    Declare Constructor (filename As String)
    Declare Destructor ()

    Declare Sub nextInstruction ()
    
    Declare Function accessGlobal (vname As String) As Double Ptr

    As uInteger ended
    As uInteger error

  Private:
    As String _filename
    As uInteger _filehandle
    As uInteger _line_number
    
    As variableSpace Ptr _global_var_space
    As variableSpace Ptr _local_var_space
    
    As Any Ptr _info_mutex
End Type

Constructor interpreter (filename As String)
  this._info_mutex = MutexCreate()
  MutexLock(this._info_mutex)
  
  this._filename = filename
  this._filehandle = FreeFile()
  Open filename For Input As #this._filehandle
  this._line_number = 1
  this.ended = 0
  this.error = NONE
  
  this._global_var_space = New variableSpace()
  this._local_var_space = this._global_var_space
  
  MutexUnlock(this._info_mutex)
End Constructor

Destructor interpreter ()
  MutexLock(this._info_mutex)
  this.ended = Not 0
  Close this._filehandle
  this._filehandle = 0
  
  this._local_var_space->destroyAllParents()
  Delete this._local_var_space
  
  MutexDestroy(this._info_mutex)
End Destructor

Sub interpreter.nextInstruction()
Dim As String curLine, tmp, thisWord, tWord
Dim As uInteger tW
Dim As Double tVal

Dim As words_list wordsList = words_list()

  If this.ended <> 0 Then Exit Sub

  If Eof(this._filehandle) Then
    this.ended = Not 0
    Exit Sub
  End If
  
  MutexLock(this._info_mutex)

  'Read the next line of input
  Line Input #this._filehandle, curLine

  'Set the string in the wordslist to the inputted line
  wordsList.setString(curLine)

  'Get the first word, make it lowercase
  thisWord = LCase(wordsList.getWord(0))
  
  'Try different things depending on its length
  Select Case Len(thisWord)
    'If it's a 3-character word...
    Case 3:
      'Check if it's END or CLS
      Select Case thisWord
        Case "end":
          this.ended = Not 0
          MutexUnlock(this._info_mutex)
          Exit Sub
        Case "cls":
          Cls
        
        Case "dim":
          'Get the first word after Dim
          tW = 1
          tWord = wordsList.getWord(1)
          
          'Continue so long as there is a comma to the right of a word
          Do
            'If the variable name is invalid, record the error but continue
            If IsNumeric(Left(tWord, 1)) Then this.error = BAD_VAR_NAME
            If IsKeyword(tWord) Then this.error = KEYWORD_VAR_NAME

            'If the variable already exists, record an error
            If this._local_var_space->accessVariable(tWord) = 0 Then this.error = DUPLICATED_DEF

            'If we find an equal sign next, get the value after the equal sign...
            If wordsList.getWord(tW+1) = "=" Then
              tW += 2
              'Get the next word
              If IsNumeric(wordsList.getWord(tW)) Then
                'If it's a number, convert it to numeric form
                tVal = Val(wordsList.getWord(tW))
              Else
                'Otherwise assume it's a variable
                tVal = this._local_var_space->getVariable(LCase(wordsList.getWord(tW)))
              End If
            Else
              tVal = 0
            End If
            
            'Create the variable with an initial value of the value specified or 0 if none is specified
            this._local_var_space->addVariable(tWord, 0)
            
            tW += 1
            
            'Get the next word
            tWord = wordsList.getWord(tW)
          Loop While ((Right(tWord, 1) <> ",") Or (wordsList.getWord(tW+1) = ",")) And tWord <> "" 
      End Select
    
    'If it's a 5-character word...
    Case 5:
      'Check if it's SLEEP or PRINT
      Select Case thisWord
        Case "sleep":
          'Variable parameters for Sleep
          Select Case wordsList.numWords
            Case 1:
              Sleep
            Case 2:
              Sleep Val(wordsList.getWord(1))
            Case 3:
              'For multiple parameters, trim off the , in between parameters
              Sleep Val(RTrim(wordsList.getWord(1), ",")), Val(wordsList.getWord(2))
          End Select
          
        Case "input":
          'Start with the first word
          tW = 1
          tWord = wordsList.getWord(1)
          
          'For each argument
          Do While tWord <> ""
            'If it ends with a ;, we trim it off and Print with ; so everything is on the same line
            If Right(tWord,1) = ";" Then
              'If it's a string...
              If Left(tWord, 1) = """" Then
                'Truncate the leading and trailing " off plus the trailing ;
                tmp = Mid(tWord, 2, Len(tWord)-3)
                Print tmp;
              'Otherwise, assume it's a variable
              Else
                'Get the variable value, removing the trailing ;
                Input tVal
                this._local_var_space->setVariable(Left(tWord,Len(tWord)-1), tVal)
              End If
            'But if it doesn't end with ;, don't trim it off
            Else
              'If it's a string...
              If Left(tWord, 1) = """" Then
                'Truncate the leading and trailing " off
                tmp = Mid(tWord, 2, wordsList.wordLength(tW)-2)
                Print tmp
              'Otherwise, assume it's a variable
              Else
                Input tVal
                this._local_var_space->setVariable(tWord, tVal)
              End If
              
              'Since it's the last parameter to print, we exit the loop
              Exit Do
            End If
            
            'Next word
            tW += 1
            tWord = wordsList.getWord(tW)
            
            'Forever, until there are no parameters left.
          Loop
          
        Case "print":
          'Start with the first word
          tW = 1
          tWord = wordsList.getWord(1)
          
          If tWord = "" Then Print ""
          
          'For each argument
          Do While tWord <> ""
            'If it ends with a ;, we trim it off and Print with ; so everything is on the same line
            If Right(tWord,1) = ";" Then
              'If it's a string...
        	    If Left(tWord, 1) = """" Then
          	    'Truncate the leading and trailing " off plus the trailing ;
                tmp = Mid(tWord, 2, Len(tWord)-3)
              'Otherwise, assume it's a variable
        	    Else
        	      'Get the variable value, removing the trailing ;
            	  tmp = Str(this._local_var_space->getVariable(Left(tWord,Len(tWord)-1)))
        	    End If
        	    Print tmp;
        	  'But if it doesn't end with ;, don't trim it off
            Else
              'If it's a string...
        	    If Left(tWord, 1) = """" Then
          	    'Truncate the leading and trailing " off
                tmp = Mid(tWord, 2, wordsList.wordLength(tW)-2)
              'Otherwise, assume it's a variable
          	  Else
            	  tmp = Str(this._local_var_space->getVariable(tWord))
              End If
              Print tmp
              
              'Since it's the last parameter to print, we exit the loop
              Exit Do
        	  End If
        	  
        	  'Next word
        	  tW += 1
        	  tWord = wordsList.getWord(tW)
        	  
        	  'Forever, until there are no parameters left.
          Loop
      End Select
  End Select
  
  MutexUnlock(this._info_mutex)
End Sub

Function interpreter.accessGlobal (vname As String) As Double Ptr
  Return this._global_var_space->accessVariable(vname)
End Function

Dim As interpreter myInterpreter = interpreter("someScript5.fbsc")
Dim As Double Ptr myVar

Do
  myInterpreter.nextInstruction()
Loop Until myInterpreter.ended <> 0

Print ""
Print "SCRIPT ENDED."

'EXAMPLE:  How to access global variables from inside your main program
myVar = myInterpreter.accessGlobal("someVar")
If myVar <> 0 Then
  Print "Final value of someVar:  " + Str(*myVar)
Else
  Print "someVar does not exist!"
End If

Sleep

End