/'*
 @file  enet.h
 @brief ENet public header file
'/

#ifndef __ENET_ENET_H__
#define __ENET_ENET_H__



#ifdef __FB_WIN32__
#include "enet/win32.bi"
#Else
	Extern "C"
		#include "enet/unix.bi"
	End extern
#EndIf

Extern "C"

#include "enet/types.bi"
#include "enet/protocol.bi"
#include "enet/list.bi"
#include "enet/callbacks.bi"

#Inclib "enet"

enum ENetVersion
	ENET_VERSION = 1
end enum


enum ENetSocketType
	ENET_SOCKET_TYPE_STREAM = 1
	ENET_SOCKET_TYPE_DATAGRAM = 2
end enum


enum ENetSocketWait
	ENET_SOCKET_WAIT_NONE = 0
	ENET_SOCKET_WAIT_SEND = (1 shl 0)
	ENET_SOCKET_WAIT_RECEIVE = (1 shl 1)
end enum


enum ENetSocketOption
	ENET_SOCKOPT_NONBLOCK = 1
	ENET_SOCKOPT_BROADCAST = 2
	ENET_SOCKOPT_RCVBUF = 3
	ENET_SOCKOPT_SNDBUF = 4
end enum


enum 
	ENET_HOST_ANY = 0
	ENET_HOST_BROADCAST_ = &hFFFFFFFF
	
	ENET_PORT_ANY = 0
end enum

/'*
 * Portable internet address structure.
 *
 * The host must be specified in network byte-order, and the port must be in host
 * byte-order. The constant ENET_HOST_ANY may be used to specify the default
 * server host. The constant ENET_HOST_BROADCAST may be used to specify the
 * broadcast address (255.255.255.255).  This makes sense for enet_host_connect,
 * but not for enet_host_create.  Once a server responds to a broadcast, the
 * address is updated from ENET_HOST_BROADCAST to the server's actual IP address.
 '/

type ENetAddress
	host as enet_uint32
	port as enet_uint16
end type

/'*
 * Packet flag bit constants.
 *
 * The host must be specified in network byte-order, and the port must be in
 * host byte-order. The constant ENET_HOST_ANY may be used to specify the
 * default server host.

   @sa ENetPacket
'/

enum ENetPacketFlag
	ENET_PACKET_FLAG_RELIABLE = (1 shl 0)
  ENET_PACKET_FLAG_UNSEQUENCED = (1 shl 1)
	ENET_PACKET_FLAG_NO_ALLOCATE = (1 shl 2)
end enum

'Type _ENetPacket As ENetPacket
'Type ENetPacketFreeCallback As Sub (As _ENetPacket ptr)

/'*
 * ENet packet structure.
 *
 * An ENet data packet that may be sent to or received from a peer. The shown
 * fields should only be read and never modified. The data field contains the
 * allocated data for the packet. The dataLength fields specifies the length
 * of the allocated data.  The flags field is either 0 (specifying no flags),
 * or a bitwise-or of any combination of the following flags:
 *
 *    ENET_PACKET_FLAG_RELIABLE - packet must be received by the target peer
 *    and resend attempts should be made until the packet is delivered
 *
 *    ENET_PACKET_FLAG_UNSEQUENCED - packet will not be sequenced with other packets
 *    (not supported for reliable packets)
 *
 *    ENET_PACKET_FLAG_NO_ALLOCATE - packet will not allocate data, and user must supply it instead

   @sa ENetPacketFlag
 '/

type ENetPacket
	referenceCount as size_t
	flags as enet_uint32
	data as enet_uint8 Ptr
	dataLength as size_t
'	freeCallback as ENetPacketFreeCallback
	freeCallback as Sub (As ENetPacket ptr)
 
end type


type ENetAcknowledgement
	acknowledgementList as ENetListNode
	sentTime as enet_uint32
	command as ENetProtocol
end type


type ENetOutgoingCommand
	outgoingCommandList as ENetListNode
	reliableSequenceNumber as enet_uint16
	unreliableSequenceNumber as enet_uint16
	sentTime as enet_uint32
	roundTripTimeout as enet_uint32
	roundTripTimeoutLimit as enet_uint32
	fragmentOffset as enet_uint32
	fragmentLength as enet_uint16
	sendAttempts as enet_uint16
	command as ENetProtocol
	packet as ENetPacket ptr
end type

type ENetIncomingCommand
	incomingCommandList as ENetListNode
	reliableSequenceNumber as enet_uint16
	unreliableSequenceNumber as enet_uint16
	command as ENetProtocol
	fragmentCount as enet_uint32
	fragmentsRemaining as enet_uint32
	fragments as enet_uint32 ptr
	packet as ENetPacket ptr
end type

enum ENetPeerState
	ENET_PEER_STATE_DISCONNECTED = 0
	ENET_PEER_STATE_CONNECTING = 1
	ENET_PEER_STATE_ACKNOWLEDGING_CONNECT = 2
	ENET_PEER_STATE_CONNECTION_PENDING = 3
	ENET_PEER_STATE_CONNECTION_SUCCEEDED = 4
	ENET_PEER_STATE_CONNECTED = 5
	ENET_PEER_STATE_DISCONNECT_LATER = 6
	ENET_PEER_STATE_DISCONNECTING = 7
	ENET_PEER_STATE_ACKNOWLEDGING_DISCONNECT = 8
	ENET_PEER_STATE_ZOMBIE = 9
end enum

#ifndef ENET_BUFFER_MAXIMUM
#define ENET_BUFFER_MAXIMUM (1 + (2 * ENET_PROTOCOL_MAXIMUM_PACKET_COMMANDS))
#EndIf

enum 
	ENET_HOST_RECEIVE_BUFFER_SIZE = 256*1024
	ENET_HOST_SEND_BUFFER_SIZE = 256*1024
	ENET_HOST_BANDWIDTH_THROTTLE_INTERVAL = 1000
	ENET_HOST_DEFAULT_MTU = 1400
	ENET_PEER_DEFAULT_ROUND_TRIP_TIME = 500
	ENET_PEER_DEFAULT_PACKET_THROTTLE = 32
	ENET_PEER_PACKET_THROTTLE_SCALE = 32
	ENET_PEER_PACKET_THROTTLE_COUNTER = 7
	ENET_PEER_PACKET_THROTTLE_ACCELERATION = 2
	ENET_PEER_PACKET_THROTTLE_DECELERATION = 2
	ENET_PEER_PACKET_THROTTLE_INTERVAL = 5000
	ENET_PEER_PACKET_LOSS_SCALE = (1 shl 16)
	ENET_PEER_PACKET_LOSS_INTERVAL = 10000
	ENET_PEER_WINDOW_SIZE_SCALE = 64*1024
	ENET_PEER_TIMEOUT_LIMIT = 32
	ENET_PEER_TIMEOUT_MINIMUM = 5000
	ENET_PEER_TIMEOUT_MAXIMUM = 30000
	ENET_PEER_PING_INTERVAL = 500
	ENET_PEER_UNSEQUENCED_WINDOWS = 64
	ENET_PEER_UNSEQUENCED_WINDOW_SIZE = 1024
	ENET_PEER_FREE_UNSEQUENCED_WINDOWS = 32
	ENET_PEER_RELIABLE_WINDOWS = 16
	ENET_PEER_RELIABLE_WINDOW_SIZE = &h1000
	ENET_PEER_FREE_RELIABLE_WINDOWS = 8
end enum

type ENetChannel
	outgoingReliableSequenceNumber as enet_uint16
	outgoingUnreliableSequenceNumber as enet_uint16
	usedReliableWindows as enet_uint16
	reliableWindows(0 to ENET_PEER_RELIABLE_WINDOWS-1) as enet_uint16
	incomingReliableSequenceNumber as enet_uint16
	incomingReliableCommands as ENetList
	incomingUnreliableCommands as ENetList
end type

/'*
 * An ENet peer which data packets may be sent or received from.
 *
 * No fields should be modified unless otherwise specified.
 '/
Type _ENetHost As EnetHost

type ENetPeer
	host as _ENetHost ptr
	outgoingPeerID as enet_uint16
	incomingPeerID as enet_uint16
	sessionID as enet_uint32
	address as ENetAddress
	data as any Ptr
	state as ENetPeerState
	channels as ENetChannel ptr
	channelCount as size_t
	incomingBandwidth as enet_uint32
	outgoingBandwidth as enet_uint32
	incomingBandwidthThrottleEpoch as enet_uint32
	outgoingBandwidthThrottleEpoch as enet_uint32
	incomingDataTotal as enet_uint32
	outgoingDataTotal as enet_uint32
	lastSendTime as enet_uint32
	lastReceiveTime as enet_uint32
	nextTimeout as enet_uint32
	earliestTimeout as enet_uint32
	packetLossEpoch as enet_uint32
	packetsSent as enet_uint32
	packetsLost as enet_uint32
	packetLoss as enet_uint32
	packetLossVariance as enet_uint32
	packetThrottle as enet_uint32
	packetThrottleLimit as enet_uint32
	packetThrottleCounter as enet_uint32
	packetThrottleEpoch as enet_uint32
	packetThrottleAcceleration as enet_uint32
	packetThrottleDeceleration as enet_uint32
	packetThrottleInterval as enet_uint32
	lastRoundTripTime as enet_uint32
	lowestRoundTripTime as enet_uint32
	lastRoundTripTimeVariance as enet_uint32
	highestRoundTripTimeVariance as enet_uint32
	roundTripTime as enet_uint32
	roundTripTimeVariance as enet_uint32
	mtu as enet_uint16
	windowSize as enet_uint32
	reliableDataInTransit as enet_uint32
	outgoingReliableSequenceNumber as enet_uint16
	acknowledgements as ENetList
	sentReliableCommands as ENetList
	sentUnreliableCommands as ENetList
	outgoingReliableCommands as ENetList
	outgoingUnreliableCommands as ENetList
	incomingUnsequencedGroup as enet_uint16
	outgoingUnsequencedGroup as enet_uint16
	unsequencedWindow(0 to (ENET_PEER_UNSEQUENCED_WINDOW_SIZE / 32)-1) as enet_uint32
	disconnectData as enet_uint32
end type

/'* An ENet host for communicating with peers.
  *
  * No fields should be modified.

    @sa enet_host_create()
    @sa enet_host_destroy()
    @sa enet_host_connect()
    @sa enet_host_service()
    @sa enet_host_flush()
    @sa enet_host_broadcast()
    @sa enet_host_bandwidth_limit()
    @sa enet_host_bandwidth_throttle()
  '/

type ENetHost
	socket as ENetSocket
	address as ENetAddress
	incomingBandwidth as enet_uint32
	outgoingBandwidth as enet_uint32
	bandwidthThrottleEpoch as enet_uint32
	mtu as enet_uint32
	recalculateBandwidthLimits as integer
	peers as ENetPeer ptr
	peerCount as size_t
	serviceTime as enet_uint32
	lastServicedPeer as ENetPeer ptr
	continueSending as integer
	packetSize as size_t
	headerFlags as enet_uint16
	commands(0 to ENET_PROTOCOL_MAXIMUM_PACKET_COMMANDS-1) as ENetProtocol
	commandCount as size_t
	buffers(0 to (1 + (2 * ENET_PROTOCOL_MAXIMUM_PACKET_COMMANDS))-1) as ENetBuffer
	bufferCount as size_t
	receivedAddress as ENetAddress
	receivedData(0 to ENET_PROTOCOL_MAXIMUM_MTU-1) as enet_uint8
	receivedDataLength as size_t
end type


/'*
 * An ENet event type, as specified in @ref ENetEvent.
 '/

enum ENetEventType
	ENET_EVENT_TYPE_NONE = 0
	ENET_EVENT_TYPE_CONNECT = 1
	ENET_EVENT_TYPE_DISCONNECT = 2
	ENET_EVENT_TYPE_RECEIVE = 3
end enum

Type ENetEvent
	type as ENetEventType
	peer as ENetPeer ptr
	channelID as enet_uint8
	data as enet_uint32
	packet as ENetPacket ptr
end type


declare function enet_initialize() as integer
declare function enet_initialize_with_callbacks ( version as ENetVersion,  inits as ENetCallbacks ptr) as integer
declare sub enet_deinitialize ()
declare function enet_time_get () as enet_uint32
declare sub enet_time_set ( as enet_uint32)
declare function enet_socket_create( as ENetSocketType,  as ENetAddress ptr) as ENetSocket
declare function enet_socket_accept( as ENetSocket,  as ENetAddress ptr) as ENetSocket
declare function enet_socket_connect( as ENetSocket,  as ENetAddress ptr) as integer
declare function enet_socket_send( as ENetSocket,  as ENetAddress ptr,  as ENetBuffer ptr,  as size_t) as integer
declare function enet_socket_receive( as ENetSocket,  as ENetAddress ptr,  as ENetBuffer ptr,  as size_t) as integer
declare function enet_socket_wait( as ENetSocket,  as enet_uint32 ptr,  as enet_uint32) as integer
declare function enet_socket_set_option( as ENetSocket,  as ENetSocketOption,  as integer) as integer
declare sub enet_socket_destroy( as ENetSocket)
declare function enet_address_set_host( address as ENetAddress ptr,  hostName as zstring ptr) as integer
declare function enet_address_get_host_ip( address as ENetAddress ptr,  hostName as zstring ptr,  nameLength as size_t) as integer
declare function enet_address_get_host( address as ENetAddress ptr,  hostName as zstring ptr,  nameLength as size_t) as integer
declare function enet_packet_create( as any ptr,  as size_t,  as enet_uint32) as ENetPacket ptr
declare sub enet_packet_destroy( as ENetPacket ptr)
declare function enet_packet_resize( as ENetPacket ptr,  as size_t) as integer
declare function enet_crc32( as ENetBuffer ptr,  as size_t) as enet_uint32
declare function enet_host_create( as ENetAddress ptr,  as size_t,  as enet_uint32,  as enet_uint32) as ENetHost ptr
declare sub enet_host_destroy( as ENetHost ptr)
declare function enet_host_connect( as ENetHost ptr,  as ENetAddress ptr,  as size_t) as ENetPeer ptr
declare function enet_host_check_events( as ENetHost ptr,  as ENetEvent ptr) as integer
declare function enet_host_service( as ENetHost ptr,  as ENetEvent ptr,  as enet_uint32) as integer
declare sub enet_host_flush( as ENetHost ptr)
declare sub enet_host_broadcast( as ENetHost ptr,  as enet_uint8,  as ENetPacket ptr)
declare sub enet_host_bandwidth_limit( as ENetHost ptr,  as enet_uint32,  as enet_uint32)
declare sub enet_host_bandwidth_throttle( as ENetHost ptr)
declare function enet_peer_send( as ENetPeer ptr,  as enet_uint8,  as ENetPacket ptr) as integer
declare function enet_peer_receive( as ENetPeer ptr,  as enet_uint8) as ENetPacket ptr
declare sub enet_peer_ping( as ENetPeer ptr)
declare sub enet_peer_reset( as ENetPeer ptr)
declare sub enet_peer_disconnect( as ENetPeer ptr,  as enet_uint32)
declare sub enet_peer_disconnect_now( as ENetPeer ptr,  as enet_uint32)
declare sub enet_peer_disconnect_later( as ENetPeer ptr,  as enet_uint32)
declare sub enet_peer_throttle_configure( as ENetPeer ptr,  as enet_uint32,  as enet_uint32,  as enet_uint32)
declare function enet_peer_throttle( as ENetPeer ptr,  as enet_uint32) as integer
declare sub enet_peer_reset_queues( as ENetPeer ptr)
declare function enet_peer_queue_outgoing_command( as ENetPeer ptr,  as ENetProtocol ptr,  as ENetPacket ptr,  as enet_uint32,  as enet_uint16) as ENetOutgoingCommand ptr
declare function enet_peer_queue_incoming_command( as ENetPeer ptr,  as ENetProtocol ptr,  as ENetPacket ptr,  as enet_uint32) as ENetIncomingCommand ptr
declare function enet_peer_queue_acknowledgement( as ENetPeer ptr,  as ENetProtocol ptr,  as enet_uint16) as ENetAcknowledgement ptr
declare function enet_protocol_command_size( as enet_uint8) as size_t
End extern
#EndIf	/'__ENET_ENET_H__'/
