''*****************************************************************************
''
''
''	Platformer tutorial 101
''
''	Chapter 03-A
''	Frame Based Movement using Delta timing
''
''  The trick is to decouple your game logic from you rendering
''  ie: Run the logic at a fixed timestep (1/60th of a second in this case)
''      while redering is done every frame.
''  This is a good technique when you want to port old code and if
''  you just hate to do physics in delta time.
''
''	Pros:  
''      Easy SpeedUp
''	    Easy SlowDown
''      Easy Pausing
''      Faster than standard delta timing (only needs delta timing in main loop)
''      Easier to manange physics
''      Porting code from 1 platform(like the DS) to another(PC) is easier 
''  Cons:
''  	There's no way to do delta-time interpolation
''		Can't think of anything more
''
''	Richard Eric M. Lope
''	http://rel.phatcode.net
''
''
''  
''*****************************************************************************
#include once "fbgfx.bi"
#include once "FBGL2D7.bi" 
#include once "FBGL2D7.bas" 
''*****************************************************************************
''
''
''   "Our Laser"
''
''
''*****************************************************************************

#define MAX_SEGMENTS 42

const as integer SCREEN_WIDTH = 640
const as integer SCREEN_HEIGHT = 480


const as integer FALSE = 0
const as integer TRUE = not FALSE
const as single PI = atn(1) * 4


Type Vector2D
	 as single x
	 as single y
End Type

Type Laser
	
	declare constructor()
	declare sub Update()
	declare sub Draw( byval LineWidth as integer, byval Colour as GLuint )
	
	as single x
	as single y
	as single Speed
	as single Dx
	as single Dy
	as single Angle
	as single Delta
	
	as integer NewTarget 
	as integer Counter
	as integer Active
	
	as Vector2D Target
	as Vector2D Segments( MAX_SEGMENTS - 1 )
	
End Type

constructor Laser()

	x = rnd * SCREEN_WIDTH
	y = rnd * SCREEN_HEIGHT
	Speed = 7.0
	Angle = 0
	Delta = PI/32.0
	Dx = cos(Angle) * Speed
	Dy = sin(Angle) * Speed
	NewTarget = TRUE
	Target.x = rnd * SCREEN_WIDTH
	Target.y = rnd * SCREEN_HEIGHT
	for i as integer = 0 to MAX_SEGMENTS - 1
		Segments( i ).x = x
		Segments( i ).y = y
	Next
	
End Constructor

'' Look ma, no DT!
sub Laser.Update()
	
	if( NewTarget ) then
		Target.x = rnd * SCREEN_WIDTH
		Target.y = rnd * SCREEN_HEIGHT
		NewTarget = FALSE
	EndIf
	
	Dx = (cos(Angle) * Speed)
	Dy = (sin(Angle) * Speed)
	
	x += Dx
	y += Dy
	
	'' set up our segments
	Segments(MAX_SEGMENTS-1).x = x
	Segments(MAX_SEGMENTS-1).y = y
  
  	for i as integer = 0 to MAX_SEGMENTS - 2
		Segments( i ).x = Segments( i + 1 ).x
		Segments( i ).y = Segments( i + 1 ).y
  	Next
  	
  	dim as single Tx = (x - Target.x)
  	dim as single Ty = (y - Target.y)
  	if( sqr( (Tx*Tx) + (Ty*Ty) ) < 64 ) then
  		NewTarget = TRUE
  	EndIf
  		
	'' Project normal
	dim as single dot = ((Dy * (Target.x - x)) - (Dx * (Target.y - y)))
	        
	if( dot > 10) then
		Angle = (Angle - Delta)
	elseif( dot < 10) then
		Angle = (Angle + Delta)
	endif
	
	
End Sub

sub Laser.Draw( byval LineWidth as integer, byval Colour as GLuint )
	
	'' Draw Laser
	for i as integer = 0 to MAX_SEGMENTS - 2
		GL2D.LineGlow(Segments( i ).x, Segments( i ).y, Segments( i + 1 ).x, Segments( i + 1 ).y, LineWidth, Colour ) 
	Next
  	
  	'' Draw Target
  	GL2D.CircleFilled( Target.x, Target.y, 15, GL2D_RGBA( 0, 0, 255, 255) )
  	
End Sub


''*****************************************************************************
'' Our main sub
''*****************************************************************************
sub main()

	randomize timer
	
	GL2D.ScreenInit( SCREEN_WIDTH, SCREEN_HEIGHT )   ''Set up GL screen
	
	const as double FIXED_TIME_STEP = 1/60		'' set it up so that logic runs at 60 fps
	
	
	Dim as Laser Laz
	
	dim as double FPS = 0
	dim as integer Frame = 0
	dim as double t = 0
	dim as double dt = 0 
	dim as double accumulator = 0
	
	dim as double Newtime = timer
	dim as double Oldtime = timer
	dim as double Seconds = 0	
	
	dt = GL2D.GetDeltaTime( FPS, timer )
	
	do
		
		
		dt = GL2D.GetDeltaTime( FPS, timer )
		
		if( multikey(FB.SC_SPACE) ) then dt /= 2.0      '' slowdown
		if( multikey(FB.SC_TAB) ) then dt *= 2.0		'' speedup
		if( multikey(FB.SC_ENTER) ) then dt = 0.0		'' pause
		
		if( dt > FIXED_TIME_STEP ) then dt = FIXED_TIME_STEP   '' limit dt so that we don't jerk around too much
		accumulator += dt
		Newtime = timer
		Seconds += 1 * dt
		Frame += 1
		
		'' Update at a fixed timestep	
		while( accumulator >= FIXED_TIME_STEP )
			
			Laz.Update()	
			accumulator -= FIXED_TIME_STEP
			
		wend
		
			
		GL2D.ClearScreen()
			
		GL2D.Begin2D()
	
			GL2D.SetBlendMode( GL2D.BLEND_GLOW ) 
			
			dim as single Intensity = abs(sin(Seconds * 8)) * 4
			for i as integer = 0 to 4
				Laz.Draw( 20 - ( i * Intensity ), GL2D_RGBA( 200, 50, 0, 255 ) )
			Next
			
			
			GL2D.SetBlendMode( GL2D.BLEND_TRANS ) 
			GL2D.PrintScale(0, 0, 1, "FPS = " & FPS)
			GL2D.PrintScale(0, 10, 1, "Logic Elapsed Seconds " & Seconds)
			GL2D.PrintScale(0, 20, 1, "Render Elapsed Seconds " & NewTime - OldTime )
			GL2D.PrintScale(0, 30, 1, "DeltaTime " & dt )
			GL2D.PrintScale(0, 40, 1, "Accumulator " & Accumulator )
			
			GL2D.PrintScale(0, 60, 1, "Hold SPACE to SlowDown" )
			GL2D.PrintScale(0, 70, 1, "Hold TAB to SpeedUp" )
			GL2D.PrintScale(0, 80, 1, "Hold ENTER to PAUSE" )
	
		GL2D.End2D()
		
		''sleep 1,1				'' unrem this if you want
		flip
		
	Loop until multikey(FB.SC_ESCAPE)


	GL2D.ShutDown()
	
End Sub

''*****************************************************************************
''*****************************************************************************


main()


end


